//
//  XLLDragCollectionView.m
//  XLLDragCollectionView
//
//  Created by 肖乐 on 2018/3/22.
//  Copyright © 2018年 IMMoveMobile. All rights reserved.
//

#import "XLLDragCollectionView.h"

//角度
#define angelToRandian(x)  ((x)/180.0*M_PI)
@interface XLLDragCollectionView ()

// 长按手势
@property (nonatomic, strong) UILongPressGestureRecognizer *longPressGesture;
// 长按的indexPath
@property (nonatomic, strong) NSIndexPath *currentIndexPath;
// 移动的cell快照
@property (nonatomic, strong) UIView *tempMoveView;
// 记录开始长按的点位
@property (nonatomic, assign) CGPoint touchPoint;

@end

@implementation XLLDragCollectionView
@dynamic dataSource, delegate;

#pragma mark - setter
- (void)setMinimumTriggerTime:(NSTimeInterval)minimumTriggerTime
{
    _minimumTriggerTime = minimumTriggerTime;
    self.longPressGesture.minimumPressDuration = minimumTriggerTime;
}

- (instancetype)initWithFrame:(CGRect)frame collectionViewLayout:(UICollectionViewLayout *)layout
{
    if (self = [super initWithFrame:frame collectionViewLayout:layout])
    {
        self.minimumTriggerTime = 1.0;
        // 增加长按手势
        [self addLongPressGesture];
    }
    return self;
}

// 添加长按手势
- (void)addLongPressGesture
{
    UILongPressGestureRecognizer *longPress = [[UILongPressGestureRecognizer alloc] initWithTarget:self action:@selector(longPressed:)];
    longPress.minimumPressDuration = self.minimumTriggerTime;
    self.longPressGesture = longPress;
    [self addGestureRecognizer:longPress];
}

// 长按手势响应
- (void)longPressed:(UILongPressGestureRecognizer *)longPressGesture
{
    if (longPressGesture.state == UIGestureRecognizerStateBegan) {
        
        [self gestureBegin:longPressGesture];
    } else if (longPressGesture.state == UIGestureRecognizerStateChanged) {
        
        [self gestureChange:longPressGesture];
    } else if (longPressGesture.state == UIGestureRecognizerStateEnded || longPressGesture.state == UIGestureRecognizerStateCancelled) {
        
        [self gestureCancelOrEnd:longPressGesture];
    }
}

// 不同手势状态下的操作
- (void)gestureBegin:(UILongPressGestureRecognizer *)longPressGesture
{
    // - (CGPoint)locationOfTouch:(NSUInteger)touchIndex inView:(UIView*)view;  触摸点相对于指定视图的位置
    // - (CGPoint)locationInView:(UIView*)view;    在指定视图的相对位置
    // 1.根据当前collectionView长按位置，获取indexPath
    CGPoint touchPoint = [longPressGesture locationOfTouch:0 inView:longPressGesture.view];
    self.touchPoint = touchPoint;
    self.currentIndexPath = [self indexPathForItemAtPoint:touchPoint];
    // 2.根据indexPath，获取UICollectionViewCell
    UICollectionViewCell *cell = [self cellForItemAtIndexPath:self.currentIndexPath];
    // 3.创建一个这个cell的快照 snapshotViewAfterScreenUpdates
    UIView *tempMoveView = [cell snapshotViewAfterScreenUpdates:NO];
    cell.hidden = YES;
    tempMoveView.frame = cell.frame;
    self.tempMoveView = tempMoveView;
    [self addSubview:tempMoveView];
    // 4.开启边缘滚动定时器
    // ..
    // 5.开始抖动
    [self beginShakeAllCell];
    // 6.KVO观察collectionView的偏移量
    // ..
    // 7.回执
}

- (void)gestureChange:(UILongPressGestureRecognizer *)longPressGesture
{
    // x轴移动的距离
    CGFloat transformX = [longPressGesture locationOfTouch:0 inView:longPressGesture.view].x - _touchPoint.x;
    // y轴移动的距离
    CGFloat transformY = [longPressGesture locationOfTouch:0 inView:longPressGesture.view].y - _touchPoint.y;
    // 得到快照的中心点通过矩阵移动之后的位置
    self.tempMoveView.center = CGPointApplyAffineTransform(self.tempMoveView.center, CGAffineTransformMakeTranslation(transformX, transformY));
    // 更新点位
    self.touchPoint = [longPressGesture locationOfTouch:0 inView:longPressGesture.view];
    [self moveCell];

}

- (void)gestureCancelOrEnd:(UILongPressGestureRecognizer *)longPressGesture
{
    // 1.获取移动的cell
    UICollectionViewCell *cell = [self cellForItemAtIndexPath:self.currentIndexPath];
    self.userInteractionEnabled = NO;
    
    [UIView animateWithDuration:0.25 animations:^{
        // 此时cell已经move到正确位置，快照可能还有偏移。所以纠正位置
        self.tempMoveView.center = cell.center;
    } completion:^(BOOL finished) {
        
        [self stopShakeAllCell];
        [self.tempMoveView removeFromSuperview];
        cell.hidden = NO;
        self.userInteractionEnabled = YES;
    }];
}

// 开始移动cell
- (void)moveCell
{
    for (UICollectionViewCell *cell in self.visibleCells) {
        
        // 过滤掉隐藏的cell
        if ([self indexPathForCell:cell] == self.currentIndexPath) continue;
        // 计算cell与快照中心点X方向距离
        CGFloat spacingX = fabs(self.tempMoveView.center.x - cell.center.x);
        // 计算cell与快照中心点Y方向的距离
        CGFloat spacingY = fabs(self.tempMoveView.center.y - cell.center.y);
        // 如果view的身位已经与这个cell距离过去超过一半了
        if (spacingX <= self.tempMoveView.bounds.size.width / 2.0 && spacingY <= self.tempMoveView.bounds.size.height)
        {
            NSIndexPath *indexPath = [self indexPathForCell:cell];
            // 更新数据源 （模型数组顺序调整）
            [self updateDataSource:indexPath];
            // cell视图位置调整
            [self moveItemAtIndexPath:self.currentIndexPath toIndexPath:indexPath];
            self.currentIndexPath = indexPath;
            break;
        }
    }
}

- (void)updateDataSource:(NSIndexPath *)indexPath
{
    if (self.dataSource && [self.dataSource respondsToSelector:@selector(dataSourceArrayOfCollectionView:)])
    {
        NSMutableArray *sourceArray = [[self.dataSource dataSourceArrayOfCollectionView:self] mutableCopy];
        // 判断数据源是单个数组 还是 多组section形式
        BOOL isMultiSection = ([self numberOfSections] > 1) || ([self numberOfSections] == 1 && [sourceArray.firstObject isKindOfClass:[NSArray class]]);
        if (isMultiSection)
        {
            // 如果里面内嵌的是数据，将里面的数据也进行深拷贝
            [sourceArray enumerateObjectsUsingBlock:^(NSArray *obj, NSUInteger idx, BOOL * _Nonnull stop) {
                
                [sourceArray replaceObjectAtIndex:idx withObject:[obj mutableCopy]];
            }];
        }
        // 如果拖动到同一组
        if (indexPath.section == self.currentIndexPath.section) {
            
            // 取出这一组的数据源
            NSMutableArray *originSection = isMultiSection?sourceArray[self.currentIndexPath.section]:sourceArray;
            // 如果当前cell的item大于起始cell的话
            if (indexPath.item > self.currentIndexPath.item)
            {
                for (NSInteger i = self.currentIndexPath.item; i < indexPath.item; i++)
                {
                    // ?
                    [originSection exchangeObjectAtIndex:i withObjectAtIndex:i+1];
                }
            } else {
                for (NSInteger i = self.currentIndexPath.item; i > indexPath.item; i--)
                {
                    [originSection exchangeObjectAtIndex:i withObjectAtIndex:i - 1];
                }
            }
        } else { // 拖到了非同组中，所以肯定是数组套数组
            
            // 起始组数据源
            NSMutableArray *originSection = sourceArray[self.currentIndexPath.section];
            // 当前组数据源
            NSMutableArray *currentSection = sourceArray[indexPath.section];
            // 将此cell的数据加入到当前组中，并将其从原始组数据源中删除
            [currentSection insertObject:originSection[self.currentIndexPath.item] atIndex:indexPath.item];
            [originSection removeObject:originSection[self.currentIndexPath.item]];
        }
        // 将重排好的数据回执，以更新外部数据源。reload的时候不至于cell又变回来
        // ..回执
        if (self.delegate && [self.delegate respondsToSelector:@selector(dragCollectionView:newDataArrayAfterMove:)])
        {
            [self.delegate dragCollectionView:self newDataArrayAfterMove:[sourceArray copy]];
        }
    }
}


// 抖动所有cell
- (void)beginShakeAllCell
{
    CAKeyframeAnimation *anim = [CAKeyframeAnimation animation];
    anim.keyPath = @"transform.rotation";
    anim.values=@[
                  @(angelToRandian(4)),
                  @(angelToRandian(-4)),
                  @(angelToRandian(4))
                  ];
    anim.repeatCount=MAXFLOAT;
    anim.duration=0.2;
    NSArray *cells = [self visibleCells];
    /******删除设置cell的晃动**************/
    NSMutableArray *lastCells = cells.mutableCopy;
    for (UICollectionViewCell *cell in lastCells) {
        // 添加动画
        if (![cell.layer animationForKey:@"shake"])
        {
            [cell.layer addAnimation:anim forKey:@"shake"];
        }
    }
    // 给快照也添加动画
    if (![self.tempMoveView.layer animationForKey:@"shake"]) {
        [self.tempMoveView.layer addAnimation:anim forKey:@"shake"];
    }
}

- (void)stopShakeAllCell
{
    for (UICollectionViewCell *cell in [self visibleCells]) {
        [cell.layer removeAllAnimations];
    }
    [self.tempMoveView.layer removeAllAnimations];
}

@end
