package com.thinkgem.jeesite.modules.user.api;

import com.google.common.collect.Maps;
import com.thinkgem.jeesite.common.baseBean.Response;
import com.thinkgem.jeesite.common.constant.ComCode;
import com.thinkgem.jeesite.common.utils.StringUtils;
import com.thinkgem.jeesite.modules.commonError.MyException;
import com.thinkgem.jeesite.modules.img.bean.ImgRequest;
import com.thinkgem.jeesite.modules.pay.entity.RateEntity;
import com.thinkgem.jeesite.modules.sys.entity.User;
import com.thinkgem.jeesite.modules.user.bean.IdentityCodeRequest;
import com.thinkgem.jeesite.modules.user.bean.UserRequest;
import com.thinkgem.jeesite.modules.user.entity.IdentityCodeEntity;
import com.thinkgem.jeesite.modules.user.entity.Level;
import com.thinkgem.jeesite.modules.user.entity.UserEntity;
import com.thinkgem.jeesite.modules.user.service.UserApiService;
import org.apache.commons.lang3.math.NumberUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

/**
 * 前端用户相关
 */
@RestController
@RequestMapping("/api/user")
public class UserApiController {
    /** log */
    private static Logger logger = LoggerFactory.getLogger(UserApiController.class);
    /** 手机号码校验 */
    private static  final Pattern regex = Pattern.compile("^(((1[3-9]))\\d{9})|(0\\d{2}-\\d{8})|(0\\d{3}-\\d{7})$");

    @Autowired
    private UserApiService userService;

    /**
     * 判断是否已经实名认证
     *
     * @param request
     * @return
     */
    @PostMapping("/checkValidation")
    public Response checkValidation(UserRequest request) {
        Response resp = new Response();
        try {
            resp.setData(userService.getUserInfo(request));
            resp.setStatus(ComCode.STATUS_CODE_2000);
            resp.setMessage(ComCode.STATUS_CODE_2000_DESC);
        } catch (Exception e) {
            resp.setStatus(ComCode.STATUS_CODE_9998);
            resp.setMessage(ComCode.STATUS_CODE_9998_DESC);
            resp.setError(e.getMessage());
        }
        return resp;
    }

    /**
     * 获取验证码
     *
     * @param request
     * @return
     */
    @PostMapping("/getIdentifyCode")
    public Response getIdentifyCode(IdentityCodeRequest request) {
        Response resp = new Response();
        try {
            // 判断业务入参是否为空
            if (StringUtils.isBlank(request.getFunction()) || StringUtils.isBlank(request.getMobile())) {
                resp.setStatus(ComCode.STATUS_CODE_3001);
                resp.setMessage(ComCode.STATUS_CODE_3001_DESC);
                return resp;
            }
            Map identifyCodeMap = new HashMap();
//            identifyCodeMap.put("indentifyCode", userService.getIdentifyCode(request));
            resp.setData(identifyCodeMap);
            resp.setStatus(ComCode.STATUS_CODE_2000);
            resp.setMessage(ComCode.STATUS_CODE_2000_DESC);
        } catch (Exception e) {
            resp.setStatus(ComCode.STATUS_CODE_9998);
            resp.setMessage(ComCode.STATUS_CODE_9998_DESC);
            resp.setError(e.getMessage());
        }
        return resp;
    }

    /**
     * 注册
     *
     * @param request
     * @return
     */
    @PostMapping("/signUp")
    public Response signUp(UserRequest request) {
        Response resp = new Response();
        try {
            // 判断业务入参是否为空
            if (StringUtils.isBlank(request.getMobile()) || StringUtils.isBlank(request.getNewPassword()) || StringUtils.isBlank(request.getFunction()) || StringUtils.isBlank(request.getRecommendedMobile()) || StringUtils.isBlank(request.getIdentityCode())) {
                resp.setStatus(ComCode.STATUS_CODE_4001);
                resp.setMessage(ComCode.STATUS_CODE_4001_DESC);
                return resp;
            }
//            // 根据手机号 + 使用场景 + 验证码查询
//            IdentityCodeEntity identityCodeEntity = userService.getIdentifyCode(request);
//            // 验证码使用场景
//            if (identityCodeEntity == null) {
//                resp.setStatus(ComCode.STATUS_CODE_4010);
//                resp.setMessage(ComCode.STATUS_CODE_4010_DESC);
//                return resp;
//            }
//            // 验证码是否过期
//            if (!userService.checkIdentifyCodeExpired(identityCodeEntity)) {
//                resp.setStatus(ComCode.STATUS_CODE_4009);
//                resp.setMessage(ComCode.STATUS_CODE_4009_DESC);
//                return resp;
//            }
//            // 查询推荐人是否存在
//            if (userService.checkRecommend(request) == null) {
//                resp.setStatus(ComCode.STATUS_CODE_4006);
//                resp.setMessage(ComCode.STATUS_CODE_4006_DESC);
//                return resp;
//            }
//            // 查询注册手机号是否存在
//            if (userService.checkMobile(request) != null) {
//                resp.setStatus(ComCode.STATUS_CODE_4007);
//                resp.setMessage(ComCode.STATUS_CODE_4007_DESC);
//                return resp;
//            }
//            // 注册手机号与推荐人手机号是否重复
//            if (request.getMobile().equals(request.getRecommendedMobile())) {
//                resp.setStatus(ComCode.STATUS_CODE_4008);
//                resp.setMessage(ComCode.STATUS_CODE_4008_DESC);
//                return resp;
//            }
            // 保存注册用户
            resp.setData(userService.signUp(request));
            resp.setStatus(ComCode.STATUS_CODE_2000);
            resp.setMessage(ComCode.STATUS_CODE_2000_DESC);
        } catch (Exception e) {
            resp.setStatus(ComCode.STATUS_CODE_9998);
            resp.setMessage(ComCode.STATUS_CODE_9998_DESC);
            resp.setError(e.getMessage());
        }
        return resp;
    }

    /**
     * 找回密码/修改密码
     *
     * @param request
     * @return
     */
    @PostMapping("/resetPassword")
    public Response resetPassword(UserRequest request) {
        Response resp = new Response();
        try {
            // 判断业务入参是否为空
            if (StringUtils.isBlank(request.getMobile()) || StringUtils.isBlank(request.getNewPassword()) || StringUtils.isBlank(request.getFunction()) || StringUtils.isBlank(request.getIdentityCode())) {
                resp.setStatus(ComCode.STATUS_CODE_4001);
                resp.setMessage(ComCode.STATUS_CODE_4001_DESC);
                return resp;
            }
            // 根据手机号 + 使用场景 + 验证码查询
            IdentityCodeEntity identityCodeEntity = userService.getIdentifyCode(request);
            // 验证码使用场景
            if (identityCodeEntity == null) {
                resp.setStatus(ComCode.STATUS_CODE_4010);
                resp.setMessage(ComCode.STATUS_CODE_4010_DESC);
                return resp;
            }
            // 验证码是否过期
            if (!userService.checkIdentifyCodeExpired(identityCodeEntity)) {
                resp.setStatus(ComCode.STATUS_CODE_4009);
                resp.setMessage(ComCode.STATUS_CODE_4009_DESC);
                return resp;
            }
            // 查询注册手机号是否存在
            if (userService.checkMobile(request) == null) {
                resp.setStatus(ComCode.STATUS_CODE_4004);
                resp.setMessage(ComCode.STATUS_CODE_4004_DESC);
                return resp;
            }
            userService.resetPassword(request);
            resp.setStatus(ComCode.STATUS_CODE_2000);
            resp.setMessage(ComCode.STATUS_CODE_2000_DESC);
        } catch (Exception e) {
            resp.setStatus(ComCode.STATUS_CODE_4005);
            resp.setMessage(ComCode.STATUS_CODE_4005_DESC);
            resp.setError(e.getMessage());
        }
        return resp;
    }

    /**
     * 登录
     *
     * @param request
     * @return
     */
    @PostMapping("/login")
    public Response login(UserRequest request) {
        Response resp = new Response();
        try {
            // 判断业务入参是否为空
            if (StringUtils.isBlank(request.getMobile()) || (StringUtils.isBlank(request.getPassword()) // 手机号 + 密码登录
                    && !(!StringUtils.isBlank(request.getFunction()) && !StringUtils.isBlank(request.getIdentityCode())))) { // 手机号 + 验证码登录
                resp.setStatus(ComCode.STATUS_CODE_4001);
                resp.setMessage(ComCode.STATUS_CODE_4001_DESC);
                return resp;
            }
            // 查询注册手机号是否存在
            if (userService.checkMobile(request) == null) {
                resp.setStatus(ComCode.STATUS_CODE_4004);
                resp.setMessage(ComCode.STATUS_CODE_4004_DESC);
                return resp;
            }
            UserEntity userEntity = null;
            if (!StringUtils.isBlank(request.getMobile()) && !StringUtils.isBlank(request.getPassword())) {
                userEntity = userService.getUserInfoByPassword(request);
                if (userEntity == null) {
                    resp.setStatus(ComCode.STATUS_CODE_4002);
                    resp.setMessage(ComCode.STATUS_CODE_4002_DESC);
                    return resp;
                }
            } else if (!StringUtils.isBlank(request.getMobile()) && !StringUtils.isBlank(request.getFunction()) && !StringUtils.isBlank(request.getIdentityCode())) {
                // 根据手机号 + 使用场景 + 验证码查询
                IdentityCodeEntity identityCodeEntity = userService.getIdentifyCode(request);
                // 验证码使用场景
                if (identityCodeEntity == null) {
                    resp.setStatus(ComCode.STATUS_CODE_4010);
                    resp.setMessage(ComCode.STATUS_CODE_4010_DESC);
                    return resp;
                }
                // 验证码是否过期
                if (!userService.checkIdentifyCodeExpired(identityCodeEntity)) {
                    resp.setStatus(ComCode.STATUS_CODE_4009);
                    resp.setMessage(ComCode.STATUS_CODE_4009_DESC);
                    return resp;
                }
                userEntity = userService.getUserInfoByIdentifyCode(request);
                if (userEntity == null) {
                    resp.setStatus(ComCode.STATUS_CODE_4003);
                    resp.setMessage(ComCode.STATUS_CODE_4003_DESC);
                    return resp;
                }
            }
            resp.setData(userService.getUserDetailInfo(userEntity));
            resp.setStatus(ComCode.STATUS_CODE_2000);
            resp.setMessage(ComCode.STATUS_CODE_2000_DESC);
        } catch (Exception e) {
            resp.setStatus(ComCode.STATUS_CODE_9998);
            resp.setMessage(ComCode.STATUS_CODE_9998_DESC);
            resp.setError(e.getMessage());
        }
        return resp;
    }

    /**
     * 获取银行卡列表
     *
     * @param request
     * @return
     */
    @PostMapping("/getCardsInfo")
    public Response getCardsInfo(UserRequest request) {
        Response resp = new Response();
        try {
            // 判断业务入参是否为空
            if (StringUtils.isBlank(request.getCardType())) {
                resp.setStatus(ComCode.STATUS_CODE_4001);
                resp.setMessage(ComCode.STATUS_CODE_4001_DESC);
                return resp;
            }
            resp.setData(userService.getCardsInfo(request));
            resp.setStatus(ComCode.STATUS_CODE_2000);
            resp.setMessage(ComCode.STATUS_CODE_2000_DESC);
        } catch (Exception e) {
            resp.setStatus(ComCode.STATUS_CODE_9998);
            resp.setMessage(ComCode.STATUS_CODE_9998_DESC);
            resp.setError(e.getMessage());
        }
        return resp;
    }

    /**
     * 移除银行卡
     *
     * @param request
     * @return
     */
    @PostMapping("/removeCards")
    public Response removeCards(UserRequest request) {
        Response resp = new Response();
        try {
            // 判断业务入参是否为空
            if (StringUtils.isBlank(request.getCardType()) || request.getCardCodes().size() == 0) {
                resp.setStatus(ComCode.STATUS_CODE_4001);
                resp.setMessage(ComCode.STATUS_CODE_4001_DESC);
                return resp;
            }
            userService.removeCards(request);
            resp.setStatus(ComCode.STATUS_CODE_2000);
            resp.setMessage(ComCode.STATUS_CODE_2000_DESC);
        } catch (Exception e) {
            resp.setStatus(ComCode.STATUS_CODE_9998);
            resp.setMessage(ComCode.STATUS_CODE_9998_DESC);
            resp.setError(e.getMessage());
        }
        return resp;
    }

    /**
     * 添加银行卡
     *
     * @param request
     * @return
     */
    @PostMapping("/addNewCard")
    public Response addNewCard(UserRequest request) {
        Response resp = new Response();
        try {
            // 判断业务入参是否为空
            if (StringUtils.isBlank(request.getSubMerchantCode()) ||
//                    StringUtils.isBlank(request.getIdentityCode()) ||
                    StringUtils.isBlank(request.getFunction()) || StringUtils.isBlank(request.getCardType()) || StringUtils.isBlank(request.getCardNo()) || StringUtils.isBlank(request.getCardMobile()) || StringUtils.isBlank(request.getBelongBank()) || StringUtils.isBlank(request.getCardBank())) {
                resp.setStatus(ComCode.STATUS_CODE_4001);
                resp.setMessage(ComCode.STATUS_CODE_4001_DESC);
                return resp;
            }
            //进行初步校验
            if(!NumberUtils.isNumber(request.getCardNo())){
                throw new MyException("卡号格式不正确");
            }
            if(!regex.matcher(request.getCardMobile()).matches()){
                throw new MyException("手机号码格式不正确");
            }
            if(request.getCardType().equals("C")){
                if (StringUtils.isBlank(request.getCvn2())|| StringUtils.isBlank(request.getExpireDate())){
                    throw new MyException("信用卡信息需补全");
                }
            }
            // 根据手机号 + 使用场景 + 验证码查询
            request.setMobile(request.getCardMobile());
//            IdentityCodeEntity identityCodeEntity = userService.getIdentifyCode(request);
//            // 验证码使用场景
//            if (identityCodeEntity == null // 未查询到
//                    || ("5".equals(request.getFunction()) && !"D".equals(request.getCardType())) // function = 储蓄卡，但cardType = 信用卡
//                    || ("6".equals(request.getFunction()) && !"C".equals(request.getCardType()))) { // function = 信用卡，但cardType = 储蓄卡
//                resp.setStatus(ComCode.STATUS_CODE_4010);
//                resp.setMessage(ComCode.STATUS_CODE_4010_DESC);
//                return resp;
//            }
//            // 验证码是否过期
//            if (!userService.checkIdentifyCodeExpired(identityCodeEntity)) {
//                resp.setStatus(ComCode.STATUS_CODE_4009);
//                resp.setMessage(ComCode.STATUS_CODE_4009_DESC);
//                return resp;
//            }
            userService.addNewCard(request);
            resp.setStatus(ComCode.STATUS_CODE_2000);
            resp.setMessage(ComCode.STATUS_CODE_2000_DESC);
        } catch (MyException e) {
            resp.setStatus(ComCode.STATUS_CODE_5001);
            resp.setMessage(ComCode.STATUS_CODE_5001_DESC);
            resp.setError(e.getMessage());
        } catch (Exception e) {
            resp.setStatus(ComCode.STATUS_CODE_9998);
            resp.setMessage(ComCode.STATUS_CODE_9998_DESC);
            resp.setError(e.getMessage());
        }
        return resp;
    }

    /**
     * 意见反馈
     *
     * @param request
     * @return
     */
    @PostMapping("/addFeedback")
    public Response addFeedback(UserRequest request) {
        Response resp = new Response();
        try {
            // 判断业务入参是否为空
            if (StringUtils.isBlank(request.getSubMerchantCode()) || StringUtils.isBlank(request.getFeedback())) {
                resp.setStatus(ComCode.STATUS_CODE_4001);
                resp.setMessage(ComCode.STATUS_CODE_4001_DESC);
                return resp;
            }
            userService.addFeedback(request);
            resp.setStatus(ComCode.STATUS_CODE_2000);
            resp.setMessage(ComCode.STATUS_CODE_2000_DESC);
        } catch (Exception e) {
            resp.setStatus(ComCode.STATUS_CODE_9998);
            resp.setMessage(ComCode.STATUS_CODE_9998_DESC);
            resp.setError(e.getMessage());
        }
        return resp;
    }

    /**
     * 关于我们
     *
     * @param request
     * @return
     */
    @PostMapping("/getAboutUsInfo")
    public Response getAboutUsInfo(UserRequest request) {
        Response resp = new Response();
        try {
            Map<String, String> map = new HashMap<>();
            map.put("aboutUs", userService.getAboutUsInfo(request).getAboutInfo());
            resp.setData(map);
            resp.setStatus(ComCode.STATUS_CODE_2000);
            resp.setMessage(ComCode.STATUS_CODE_2000_DESC);
        } catch (Exception e) {
            resp.setStatus(ComCode.STATUS_CODE_9998);
            resp.setMessage(ComCode.STATUS_CODE_9998_DESC);
            resp.setError(e.getMessage());
        }
        return resp;
    }

    /**
     * 实名认证
     *
     * @param request
     * @return
     */
    @PostMapping("/certification")
    public Response certification(UserRequest request) {
        Response resp = new Response();
        try {
            // 判断业务入参是否为空
            if (StringUtils.isBlank(request.getSubMerchantCode())
                    || StringUtils.isBlank(request.getFunction()) || StringUtils.isBlank(request.getIdentityCode())
                    || StringUtils.isBlank(request.getRealName()) || StringUtils.isBlank(request.getIdCard())
                    || StringUtils.isBlank(request.getCardNo()) || StringUtils.isBlank(request.getCardMobile())
                    || StringUtils.isBlank(request.getBelongBank())
                    || StringUtils.isBlank(request.getCardBank())
                        ) {
                logger.debug("业务参数就异常了");
                logger.debug(request.toString());
//                    || StringUtils.isBlank(request.getIdCardPlace())|| StringUtils.isBlank(request.getIdCardFront())|| StringUtils.isBlank(request.getIdCardBack())|| StringUtils.isBlank(request.getIdCardBack())
//                    || StringUtils.isBlank(request.getIdCardHold())|| StringUtils.isBlank(request.getDebitCard())|| StringUtils.isBlank(request.getCardSrc())|| StringUtils.isBlank(request.getCardPlace())
                resp.setStatus(ComCode.STATUS_CODE_4001);
                resp.setMessage(ComCode.STATUS_CODE_4001_DESC);
                return resp;
            }
            //身份证参数格式校验
            if(!verifyIdentityCard(request.getIdCard().trim())){
                logger.debug("身份证这里校验出问题了");
                throw new MyException("身份证号码格式不正确");
            }
            //进行卡片校验
            if(!NumberUtils.isNumber(request.getCardNo().trim())){
                throw new MyException("卡号格式不正确");
            }
            //手机号码校验
            if(!regex.matcher(request.getCardMobile().trim()).matches()){
                throw new MyException("手机号码格式不正确");
            }

//            // 根据手机号 + 使用场景 + 验证码查询
//            request.setMobile(request.getCardMobile());
//            IdentityCodeEntity identityCodeEntity = userService.getIdentifyCode(request);
//            // 验证码使用场景
//            if (identityCodeEntity == null) {
//                resp.setStatus(ComCode.STATUS_CODE_4010);
//                resp.setMessage(ComCode.STATUS_CODE_4010_DESC);
//                return resp;
//            }
//            // 验证码是否过期
//            if (!userService.checkIdentifyCodeExpired(identityCodeEntity)) {
//                resp.setStatus(ComCode.STATUS_CODE_4009);
//                resp.setMessage(ComCode.STATUS_CODE_4009_DESC);
//                return resp;
//            }
            userService.certification(request);
            resp.setStatus(ComCode.STATUS_CODE_2000);
            resp.setMessage(ComCode.STATUS_CODE_2000_DESC);
        } catch (Exception e) {
            resp.setStatus(ComCode.STATUS_CODE_9998);
            resp.setMessage(ComCode.STATUS_CODE_9998_DESC);
            resp.setError(e.getMessage());
        }
        return resp;
    }

    /**
     * 获取省市
     *
     * @param request
     * @return
     */
    @PostMapping("/provinceCity")
    public Response getProvinceCity(UserRequest request) {
        Response resp = new Response();
        try {
            // 判断业务入参是否为空
            if (StringUtils.isBlank(request.getType()) || StringUtils.isBlank(request.getPid())) {
                resp.setStatus(ComCode.STATUS_CODE_4001);
                resp.setMessage(ComCode.STATUS_CODE_4001_DESC);
                return resp;
            }
            resp.setData(userService.getProvinceCity(request));
            resp.setStatus(ComCode.STATUS_CODE_2000);
            resp.setMessage(ComCode.STATUS_CODE_2000_DESC);
        } catch (Exception e) {
            resp.setStatus(ComCode.STATUS_CODE_9998);
            resp.setMessage(ComCode.STATUS_CODE_9998_DESC);
            resp.setError(e.getMessage());
        }
        return resp;
    }

    /**
     * 获取所属银行及联行号
     *
     * @param request
     * @return
     */
    @PostMapping("/belongBank")
    public Response getBelongBank(UserRequest request) {
        Response resp = new Response();
        try {
            resp.setData(userService.getBelongBank(request));
            resp.setStatus(ComCode.STATUS_CODE_2000);
            resp.setMessage(ComCode.STATUS_CODE_2000_DESC);
        } catch (Exception e) {
            resp.setStatus(ComCode.STATUS_CODE_9998);
            resp.setMessage(ComCode.STATUS_CODE_9998_DESC);
            resp.setError(e.getMessage());
        }
        return resp;
    }

    /**
     * 切换结算卡
     *
     * @param request
     * @return
     */
    @PostMapping("/changeDebit")
    public Response changeDebit(UserRequest request) {
        Response resp = new Response();
        try {
            // 判断业务入参是否为空
            if (StringUtils.isBlank(request.getSubMerchantCode()) || StringUtils.isBlank(request.getDebitCardCode())) {
                resp.setStatus(ComCode.STATUS_CODE_4001);
                resp.setMessage(ComCode.STATUS_CODE_4001_DESC);
                return resp;
            }
            userService.changeDebit(request);
            resp.setStatus(ComCode.STATUS_CODE_2000);
            resp.setMessage(ComCode.STATUS_CODE_2000_DESC);
        } catch (Exception e) {
            resp.setStatus(ComCode.STATUS_CODE_9998);
            resp.setMessage(ComCode.STATUS_CODE_9998_DESC);
            resp.setError(e.getMessage());
        }
        return resp;
    }

    /**
     * 修改用户信息
     *
     * @param request
     * @return
     */
    @PostMapping("/modifyUser")
    public Response modifyUser(UserRequest request){
        Response resp = new Response();
        if(request == null || StringUtils.isBlank(request.getSubMerchantCode())){
            resp.setStatus(ComCode.STATUS_CODE_4001);
            resp.setMessage(ComCode.STATUS_CODE_4001_DESC);
            return resp;
        }
        try {
            userService.updateUser(request);
            resp.setStatus(ComCode.STATUS_CODE_2000);
            resp.setMessage(ComCode.STATUS_CODE_2000_DESC);
        }catch (Exception e){
            resp.setStatus(ComCode.STATUS_CODE_9998);
            resp.setMessage(ComCode.STATUS_CODE_9998_DESC);
            resp.setError(e.getMessage());
        }
        return resp;
    }

    /**
     * 获取汇率信息
     *
     * @param request
     * @return
     */
    @PostMapping("/getRate")
    public Response getRate(UserRequest request){
        Response resp = new Response();
        if(request == null || StringUtils.isBlank(request.getLevel())){
            resp.setStatus(ComCode.STATUS_CODE_4001);
            resp.setMessage(ComCode.STATUS_CODE_4001_DESC);
            return resp;
        }
        try {
            Level level = userService.getLevelByUserLevel(request);
            if(level == null){
                level = new Level();
            }
            RateEntity rateEntityParam = new RateEntity();
            rateEntityParam.setUserLevel(request.getLevel());
            level.setRateList(userService.getRateListByLevel(rateEntityParam));
            resp.setData(level);
            resp.setStatus(ComCode.STATUS_CODE_2000);
            resp.setMessage(ComCode.STATUS_CODE_2000_DESC);
        }catch (Exception e){
            resp.setStatus(ComCode.STATUS_CODE_9998);
            resp.setMessage(ComCode.STATUS_CODE_9998_DESC);
            resp.setError(e.getMessage());
        }
        return resp;
    }

//    /**
//     * 修改用户等级
//     *
//     * @param request
//     * @return
//     */
//    @PostMapping("/modifyUserLevel")
//    public Response modifyUserLevel(UserRequest request){
//        Response resp = new Response();
//        if(request == null || StringUtils.isBlank(request.getSubMerchantCode())){
//            resp.setStatus(ComCode.STATUS_CODE_4001);
//            resp.setMessage(ComCode.STATUS_CODE_4001_DESC);
//            return resp;
//        }
//        try {
//            userService.updateUser(request);
//            resp.setStatus(ComCode.STATUS_CODE_2000);
//            resp.setMessage(ComCode.STATUS_CODE_2000_DESC);
//        }catch (Exception e){
//            resp.setStatus(ComCode.STATUS_CODE_9998);
//            resp.setMessage(ComCode.STATUS_CODE_9998_DESC);
//            resp.setError(e.getMessage());
//        }
//        return resp;
//    }

    /**
     * 获取邀请好友信息
     * @param request
     * @return
     */
    @PostMapping("/getMyInviting")
    public Response getMyInviting(UserRequest request){
        Response resp = new Response();
        if(request == null || StringUtils.isBlank(request.getSubMerchantCode())){
            resp.setStatus(ComCode.STATUS_CODE_4002);
            resp.setMessage(ComCode.STATUS_CODE_4001_DESC);
            return resp;
        }
        try{
            List<UserEntity> myInviting = userService.getMyInviting(request);
            Integer activeNum = userService.getActiveNum(request);
            HashMap<String, Object> map = Maps.newHashMap();
            map.put("allNum",myInviting.size());
            map.put("activeNum",activeNum);
            map.put("list",myInviting);
            resp.setData(map);
            resp.setStatus(ComCode.STATUS_CODE_2000);
            resp.setMessage(ComCode.STATUS_CODE_2000_DESC);
        }catch (Exception e){
            resp.setStatus(ComCode.STATUS_CODE_9998);
            resp.setMessage(ComCode.STATUS_CODE_9998_DESC);
            resp.setError(e.getMessage());
        }
        return resp;
    }

    private boolean verifyIdentityCard(String num) {
        String reg = "^\\d{15}$|^\\d{17}[0-9Xx]$";
        if (!num.matches(reg)) {
            return false;
        }
        return true;
    }

}
