/**
 * Copyright &copy; 2012-2016 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.thinkgem.jeesite.modules.img.util;

import com.google.common.collect.Lists;
import com.thinkgem.jeesite.common.utils.Encodes;
import com.thinkgem.jeesite.common.utils.StringUtils;
import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipFile;
import org.apache.tools.zip.ZipOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sun.misc.BASE64Decoder;
import sun.misc.BASE64Encoder;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.util.Enumeration;
import java.util.List;

/**
 * 图片上传工具类
 */
public class ImgUtils {

    private static Logger logger = LoggerFactory.getLogger(ImgUtils.class);

    /**
     * @param imgStr base64编码字符串
     * @param path   图片路径-具体到文件
     * @return
     * @Description: 将base64编码字符串转换为图片
     */
    public static boolean generateImage(String imgStr, String path) {
        if (imgStr == null)
            return false;
        BASE64Decoder decoder = new BASE64Decoder();
        try {
            byte[] b = decoder.decodeBuffer(imgStr);// 解密
            for (int i = 0; i < b.length; ++i) {// 处理数据
                if (b[i] < 0) {
                    b[i] += 256;
                }
            }
            OutputStream out = new FileOutputStream(path);
            out.write(b);
            out.flush();
            out.close();
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * @return
     * @Description: 根据图片地址转换为base64编码字符串
     */
    public static String getImageStr(String imgFile) {
        InputStream inputStream = null;
        byte[] data = null;
        try {
            inputStream = new FileInputStream(imgFile);
            data = new byte[inputStream.available()];
            inputStream.read(data);
            inputStream.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
        // 加密
        BASE64Encoder encoder = new BASE64Encoder();
        return encoder.encode(data);
    }

    public static void main(String[] args) {
        String strImg = "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";
        generateImage(strImg, "E:/1bak.jpg");
    }
}
