package com.thinkgem.jeesite.modules.user.service;

import com.thinkgem.jeesite.common.baseBean.Request;
import com.thinkgem.jeesite.common.config.Global;
import com.thinkgem.jeesite.common.constant.ComCode;
import com.thinkgem.jeesite.common.utils.DateUtils;
import com.thinkgem.jeesite.common.utils.IdGen;
import com.thinkgem.jeesite.common.utils.MD5;
import com.thinkgem.jeesite.modules.user.bean.IdentityCodeRequest;
import com.thinkgem.jeesite.modules.user.bean.UserRequest;
import com.thinkgem.jeesite.modules.user.dao.UserApiDao;
import com.thinkgem.jeesite.modules.user.entity.*;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.text.ParseException;
import java.util.Date;
import java.util.List;


/**
 * 前端用户相关
 */
@Service
@Transactional(readOnly = true)
public class UserApiService {

    @Autowired
    private UserApiDao userDao;

    public UserEntity getUserInfo(UserRequest request) {
        return userDao.getUserInfo(request);
    }

    @Transactional(readOnly = false)
    public String getIdentifyCode(IdentityCodeRequest request) {
        // 生成验证码
        String identifyCode = IdGen.randomBase62(6).toUpperCase();
        // TODO 发送验证码
        // 验证码入库
        IdentityCodeEntity identityCodeEntity = new IdentityCodeEntity(IdGen.randomBase62(64), request.getFunction(), request.getMobile(), identifyCode);
        userDao.saveIdentifyCode(identityCodeEntity);
        // 返回验证码
        return identifyCode;
    }

    public IdentityCodeEntity getIdentifyCode(UserRequest request) {
        return userDao.getIdentifyCode(request);
    }


    public boolean checkIdentifyCodeExpired(IdentityCodeEntity identityCodeEntity) throws ParseException {
        Date now = new Date();
        Date created = ComCode.YYYYMMDD_HHMMSS.parse(identityCodeEntity.getCreated());
        // 验证码半小时内有效
        if ((now.getTime() - created.getTime()) < (30 * ComCode.seconds * ComCode.mills)) {
            return true;
        }
        return false;
    }

    public UserEntity checkRecommend(UserRequest request) {
        return userDao.checkRecommend(request);
    }

    public UserEntity checkMobile(UserRequest request) {
        return userDao.checkMobile(request);
    }

    @Transactional(readOnly = false)
    public UserEntity signUp(UserRequest request) {
        // 申请子商户号
        String subMerchantId = "subMerchantId" + DateUtils.formatDateTime(new Date());
        // 保存用户信息
        String code = IdGen.randomBase62(64);
        UserEntity recommend = userDao.checkRecommend(request);
        UserEntity userEntity = new UserEntity(code, subMerchantId, request.getMobile(), MD5.sign(request.getNewPassword(), ComCode.signCode, "utf-8"), request.getRecommendedMobile(), recommend.getRecommendedPath() + recommend.getId() + ",", recommend.getBelongTo());
        userDao.signUp(userEntity);
        return userDao.getUserInfo(new Request(code));
    }

    @Transactional(readOnly = false)
    public void resetPassword(UserRequest request) {
        // 根据手机号查询用户信息，因为Controller层判断过用户信息，因此此处一定可以查到用户信息
        UserEntity userEntity = userDao.checkMobile(request);
        userEntity.setPassword(MD5.sign(request.getNewPassword(), ComCode.signCode, "utf-8"));
        // 重置密码
        userDao.resetPassword(userEntity);
    }

    public UserEntity getUserInfoByPassword(UserRequest request) {
        request.setPassword(MD5.sign(request.getPassword(), ComCode.signCode, "utf-8"));
        return userDao.getUserInfoByPassword(request);
    }

    public UserEntity getUserInfoByIdentifyCode(UserRequest request) {
        // Controller层判断过验证码信息，因此直接根据手机号查询用户信息即可
        return userDao.checkMobile(request);
    }

    public UserEntity getUserDetailInfo(UserEntity userEntity) {
        // 根据用户code获取用户详情
        UserDetailEntity userDetailEntity = userDao.getUserDetailByCode(userEntity);
        userEntity.setUserDetail(userDetailEntity);
        // 获取用户结算卡信息
        CardEntity cardEntity = userDao.getCardInfoByCode(userEntity);
        userEntity.getUserDetail().setCard(cardEntity);
        return userEntity;
    }

    public List<CardEntity> getCardsInfo(UserRequest request) {
        return userDao.getCardsInfo(request);
    }

    @Transactional(readOnly = false)
    public void removeCards(UserRequest request) {
        userDao.removeCards(request);
    }

    @Transactional(readOnly = false)
    public void addNewCard(UserRequest request) {
        String code = IdGen.randomBase62(64);
        CardEntity cardEntity = new CardEntity(code, request.getSubMerchantCode(), request.getCardType(), request.getCardNo(), request.getCardMobile(), request.getBelongBank(), request.getCardBank(), request.getCardPlace(), "N");
        userDao.addNewCard(cardEntity);
    }

    @Transactional(readOnly = false)
    public void addFeedback(UserRequest request) {
        String code = IdGen.randomBase62(64);
        FeedbackEntity feedbackEntity = new FeedbackEntity(code, request.getSubMerchantCode(), request.getFeedback());
        userDao.addFeedback(feedbackEntity);
    }

    public AboutUsEntity getAboutUsInfo(UserRequest request) {
        return userDao.getAboutUsInfo(request);
    }

    @Transactional(readOnly = false)
    public void certification(UserRequest request) {
        // 更新用户详情
        UserDetailEntity userDetailEntity = new UserDetailEntity(IdGen.randomBase62(64), request.getSubMerchantCode(), request.getRealName(), request.getIdCard(), request.getIdCardPlace(), request.getIdCardFront().replace(Global.getConfig("sys.pre.url"), ""), request.getIdCardBack().replace(Global.getConfig("sys.pre.url"), ""), request.getIdCardHold().replace(Global.getConfig("sys.pre.url"), ""), request.getDebitCard().replace(Global.getConfig("sys.pre.url"), ""));
        userDao.saveUserDetail(userDetailEntity);
        // 更新银行卡详情
        CardEntity cardEntity = new CardEntity(IdGen.randomBase62(64), request.getSubMerchantCode(), "D", request.getCardNo(), request.getCardMobile(), request.getBelongBank(), request.getCardBank(), request.getCardPlace(), "Y");
        userDao.addNewCard(cardEntity);
        // 更新用户状态
        UserEntity userEntity = new UserEntity();
        userEntity.setCode(request.getSubMerchantCode());
        userEntity.setStatus("1");
        userDao.updateUserInfo(userEntity);
    }
}
